using System;
using System.Reflection;
using System.Collections;
using System.Data;
using System.Management; // CR 2320

using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL;
using BrRules = gov.va.med.vbecs.Common.VbecsBrokenRules.VbecsUser;
using VBECSUSER = gov.va.med.vbecs.Common.VbecsTables.VbecsUser;

namespace gov.va.med.vbecs.BOL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Luke Meyer</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>8/23/2002</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	/// <summary>
	/// Represents VBECS user. Provides means to retrieve/save user information and validate user access.
	/// </summary>

	#endregion

	public class VbecsUser : BaseBusinessObject, IRecognizableByID
	{
		/// <summary>
		/// Standard comparer used to compare VBECS users by ID.
		/// </summary>
		private class VbecsUserIdComparer : BusinessObjectPropertyComparerBase
		{
			protected override int CompareProperties( object x, object y )
			{
				return Comparer.DefaultInvariant.Compare( ((VbecsUser)x).UserNTLoginID, ((VbecsUser)y).UserNTLoginID );
			}
		}

		private System.Guid _guid;
		private string _ntLoginName;
		private string _ntName;
		private string _vistaName;		
		private string _duz;
		private string _email;
		private string _initials;		
		private bool _isActive;
		private bool _isDefaultUser = false;
		
		private VbecsUserDivisionRolesCollection _userDivisionRoles;
		private VbecsUserDivisionRole _currentDivisionRole;
		private DataTable _vbecsUserPermissions;

		private static VbecsUser _loggedOnUser;

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6614"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>An instance of the class.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6615"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Default constructor used to initialize fresh, empty instance of the user.
		/// </summary>
		public VbecsUser() : base()
		{
			SetInitialBrokenRules( BrRules.FormClassName, BrRules.RuleSets.BaseRuleSet );
			
			_guid = Guid.NewGuid();
			
			// explicit property assignments are needed to break rules. 
			UserNTLoginID = null;			
			UserDuz = null;
			UserName = null;
			UserInitials = null;
			VistAUserName = null;
			UserEmail = null;
			IsActive = true;

			_vbecsUserPermissions = null;

			IsNew = true;
			IsDirty = true;
		}

		/// <summary>
		/// Creates an instance of the class from a given <see cref="DataRow"/>.
		/// </summary>
		/// <param name="dtRow">Source <see cref="DataRow"/> to load data from.</param>
		private VbecsUser( DataRow dtRow ) : this()
		{
			LoadFromDataRow( dtRow );
		}	

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/23/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="269"> 
		///		<ExpectedInput>Valid login ID.</ExpectedInput>
		///		<ExpectedOutput>Valid VbecsUser object.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2357"> 
		///		<ExpectedInput>Null login ID.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///	<summary>
		/// Constructor. Retrieves user information from the database based on user ID.
		/// </summary>
		public VbecsUser( string loginID, bool userMustExist ) : this()
		{
			System.Data.DataTable dtUser = DAL.VbecsUser.GetRequireVbecsUser(loginID, false);

			this._isDefaultUser = false;

			if (dtUser.Rows.Count == 0)
			{
				if (userMustExist)
				{
					throw new BOL.BusinessObjectException(Common.StrRes.SysErrMsg.UC018.UserNotFound().ResString);
				}
				else		//UserID not found in table -- possible conversion/manual update error
				{
					//Use our default VBECS user 
					dtUser = DAL.VbecsUser.GetRequireVbecsUser("VBECS", true);
					this._isDefaultUser = true;
				}
			}

			this.LoadFromDataRow(dtUser.Rows[0]);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/20/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7256"> 
		///		<ExpectedInput>Valid userID</ExpectedInput>
		///		<ExpectedOutput>VbecsUser</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7257"> 
		///		<ExpectedInput>Invalid UserId</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///	<summary>
		/// Constructor. Retrieves user information from the database based on user ID.
		/// </summary>
		public VbecsUser( string loginID ) : this(loginID, true)
		{
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6943"> 
		///		<ExpectedInput>Object is altered and it's marked as dirty.</ExpectedInput>
		///		<ExpectedOutput>Object is returned to clean state.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6944"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Reloads user data from DB cancelling all changes. 
		/// </summary>
		public void RefreshFromDb()
		{
			if( this.IsNew )
				throw( new BusinessObjectException( StrRes.SysErrMsg.Common.NewBusinessObjectCannotBeRestoredFromDb( typeof(VbecsUser).Name ).ResString ) );

			LoadFromDataRow( DAL.VbecsUser.GetRequireVbecsUser( this.UserNTLoginID,this._isDefaultUser ).Rows[0]);
			LoadUserDivisionRoles();
			
			// This is not really required but for the sake of safety.
			_vbecsUserPermissions = null;
		}
		
		private void LoadUserDivisionRoles()
		{
			_userDivisionRoles = !this.IsNew ? new VbecsUserDivisionRolesCollection( DAL.VbecsUser.GetVbecsUserDivisionsAndRoles( this.UserNTLoginID ) ) : new VbecsUserDivisionRolesCollection();
		}

		private void LoadUserSystemWidePermissions()
		{
			_vbecsUserPermissions = DAL.VbecsUser.GetVbecsUserPermissions( UserNTLoginID );
		}

		private void OnPropertyChanged()
		{
			IsDirty = true;
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="1390"> 
		///		<ExpectedInput>User object is valid, valid data row.</ExpectedInput>
		///		<ExpectedOutput>Valid data row loaded with user data.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1389"> 
		///		<ExpectedInput>Invalid data row.</ExpectedInput>
		///		<ExpectedOutput>ArgumentException.</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="6945"> 
		///		<ExpectedInput>Null</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6946"> 
		///		<ExpectedInput>User object is invalid.</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///  Loads a <see cref="DataRow"/> from an object.
		/// </summary>
		public override DataRow LoadDataRowFromThis( DataRow dtRow )
		{
			if( dtRow == null )
				throw( new ArgumentNullException( "dtRow" ) );

			if( !IsValid )
				throw( new BusinessObjectException( StrRes.SysErrMsg.Common.UnableToSaveInvalidBusinessObject( typeof( VbecsUser ).Name ).ResString ) );

			RequireNullableColumns( dtRow );
			Common.Utility.RequireColumns( dtRow, GetNonNullableColumns() );

			dtRow[ VBECSUSER.VbecsUserId ] = StDbNullConvert.From( UserNTLoginID );
			dtRow[ VBECSUSER.VbecsUserGuid ] = StDbNullConvert.From( _guid );
			dtRow[ VBECSUSER.UserDUZ ] = StDbNullConvert.From( UserDuz );
			dtRow[ VBECSUSER.UserName ] = StDbNullConvert.From( UserName );
			dtRow[ VBECSUSER.UserInitials ] = StDbNullConvert.From( UserInitials );
			dtRow[ VBECSUSER.VistaUserName ] = StDbNullConvert.From( VistAUserName );			
			dtRow[ VBECSUSER.UserEmail ] = StDbNullConvert.From( UserEmail );			

			dtRow[ VBECSUSER.RecordStatusCode ] = Utility.GetRecordStatusCodeCharFromBoolean( IsActive );			
			dtRow[ VBECSUSER.RowVersion ] = StDbNullConvert.From( RowVersion );

			return dtRow;
		}

		/// <summary>
		/// Loads an object from a <see cref="DataRow"/>
		/// </summary>
		/// <param name="dtRow">Source <see cref="DataRow"/> to load an object instance from.</param>
		protected override void LoadFromDataRow( DataRow dtRow )
		{
			if( dtRow == null )
				throw( new ArgumentNullException( "dtRow" ) );
						
			RequireNullableColumns( dtRow );
			Common.Utility.RequireNonNullColumns( dtRow, GetNonNullableColumns() );
			
			UserNTLoginID = StDbNullConvert.ToString( dtRow[ VBECSUSER.VbecsUserId ] );			
			_guid = StDbNullConvert.ToGuid( dtRow[ VBECSUSER.VbecsUserGuid ] );
			UserDuz = StDbNullConvert.ToString( dtRow[ VBECSUSER.UserDUZ ] );			
			UserName = StDbNullConvert.ToString( dtRow[ VBECSUSER.UserName ] );	
			UserInitials = StDbNullConvert.ToString( dtRow[ VBECSUSER.UserInitials ] );
			VistAUserName = StDbNullConvert.ToString( dtRow[ VBECSUSER.VistaUserName ] );
			UserEmail = StDbNullConvert.ToString( dtRow[ VBECSUSER.UserEmail ] );						

			IsActive = Utility.GetIsActiveStatusByStatusCode( dtRow[ VBECSUSER.RecordStatusCode ] );
			RowVersion = StDbNullConvert.ToRowversion( dtRow[ VBECSUSER.RowVersion ] );

			if( !IsValid )
				throw( new BusinessObjectException( StrRes.SysErrMsg.Common.InvalidDataLoadedFromDataRowIntoBusinessObject( typeof( VbecsUser ).Name, MethodBase.GetCurrentMethod().Name ).ResString ) );

			IsNew = false;
			IsDirty = false;
		}

		private void RequireNullableColumns( DataRow dtRow )
		{
			Common.Utility.RequireColumns( dtRow, VBECSUSER.UserEmail );
		}

		private string[] GetNonNullableColumns()
		{
			return new string[] { VBECSUSER.VbecsUserGuid, VBECSUSER.VbecsUserId, VBECSUSER.UserDUZ, VBECSUSER.UserName, VBECSUSER.VistaUserName, VBECSUSER.UserInitials, VBECSUSER.RecordStatusCode, VBECSUSER.RowVersion };
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/15/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6675"> 
		///		<ExpectedInput>Current user running unit tests has access to VBECS, permission to check - standard access.</ExpectedInput>
		///		<ExpectedOutput>User is authorized for standard access.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6947"> 
		///		<ExpectedInput>Valid logged on user with techologist access to the test division, permission to check - define VBECS users.</ExpectedInput>
		///		<ExpectedOutput>User is not authorized to define VBECS users.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6676"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Checks whether logged on user is authorized to perform a specific function (has a specific permission).
		/// Implements BR_6.02, PT_19.01. Not logged on user is not authorized for anything.
		/// </summary>
		/// <param name="function">Function to check against.</param>
		/// <returns>True if user is authorized, otherwise - false.</returns>
		public bool IsAuthorizedForFunction( FunctionCode function )
		{
			return PermissionsTable.Select( String.Concat( VbecsTables.RoleFunction.RoleFunctionId, " = ", ((int)function).ToString(), " and ", GetUserIdAndDivisionPermissionsCheckClause() ) ).Length > 0;
		}

		private string GetUserIdAndDivisionPermissionsCheckClause()
		{
			if( CurrentDivisionRole == null )
				return "1 = 0"; // Safeguard check, user that is not logged on is not authorized for anything.

			return String.Concat( VBECSUSER.VbecsUserId, " = '", UserNTLoginID, "' and ", VbecsTables.VamcDivision.DivisionCode," = '", CurrentDivisionRole.Division.DivisionCode, "'" );
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/15/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6677"> 
		///		<ExpectedInput>Logged on user with technologist role in test division.</ExpectedInput>
		///		<ExpectedOutput>User is authorized for the technologist role.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6948"> 
		///		<ExpectedInput>Logged on user with technologist role in test division.</ExpectedInput>
		///		<ExpectedOutput>User is not authorized for the administrator/supervisor role.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6678"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Checks whether user is authorized to to perform the specified role.
		/// </summary>
		/// <param name="requestedRole">User role to check against.</param>
		/// <returns>True if user is authorized, otherwise - false.</returns>
		public bool IsAuthorized( UserRoleType requestedRole ) 
		{
			if( CurrentDivisionRole == null ) // Safeguard check
				return false;

			return (int)CurrentDivisionRole.Role.UserRoleId >= (int)requestedRole;
		}

		/// <summary>
		/// Gets new template <see cref="DataTable"/> used for interlayer 
		/// VBECS user data exchange.
		/// </summary>
		/// <returns>VBECS user template <see cref="DataTable"/>.</returns>
		protected virtual DataTable GetUserTemplateDataTable()
		{
			return DAL.VbecsUser.GetUserTemplateDataTable();
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7034"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7035"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Saves user in the database. Inserts record for new user and updates for existing one.
		/// </summary>
		/// <param name="lastUpdateFunction">Last update function.</param>
		public void Persist( UpdateFunction lastUpdateFunction )
		{	
			ArrayList userData = new ArrayList( 1 );
			userData.Add( this );

			try
			{
				// CR1902, CR2597
				if (this.DivisionsAndRoles.IsDirty && !VerifyUsersDivisions())
				// CR2597 end
				{
					RuleBroken( BrRules.DivisionDuzMismatch, true );
				}
				else
				{
					// CR1902 end
					PersistUsers( userData, lastUpdateFunction );
				}
			}
			catch( Common.UniqueDatabaseConstraintViolatedException xcp )
			{
				if( !SetPostSaveBrokenRulesBasedOnException( xcp ) ) 
					throw;  // some unexpected conflict
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/15/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7967"> 
		///		<ExpectedInput>Valid user list</ExpectedInput>
		///		<ExpectedOutput>User saved</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7968"> 
		///		<ExpectedInput>Null user list</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// PersistUsers
		/// </summary>
		/// <param name="usersToPersist"></param>
		/// <param name="lastUpdateFunction"></param>
		public static void PersistUsers( IList usersToPersist, UpdateFunction lastUpdateFunction )
		{
			if( usersToPersist == null )
				throw( new ArgumentNullException( "usersToPersist" ) );

			DataTable usersCoreData;
			IList usersRolesData;

			ExportUsersDataForSave( usersToPersist, out usersCoreData, out usersRolesData );

			DAL.VbecsUser.PersistUsers( usersCoreData, usersRolesData, lastUpdateFunction );

			foreach( VbecsUser user in usersToPersist )
				user.DoSuccessfulSavePostActions();
		}

		private void DoSuccessfulSavePostActions()
		{
			this.IsNew = false; // need to reset this, or the next line will fail
			this.RefreshFromDb();
		}

		/// <summary>
		/// ExportUsersDataForSave
		/// </summary>
		/// <param name="usersToPersist"></param>
		/// <param name="usersCoreData"></param>
		/// <param name="usersRolesDataTablesList"></param>
		internal static void ExportUsersDataForSave( IList usersToPersist, out DataTable usersCoreData, out IList usersRolesDataTablesList )
		{
			if( usersToPersist == null )
				throw( new ArgumentNullException( "usersToPersist" ) );

			usersCoreData = DAL.VbecsUser.GetUserTemplateDataTable();
			usersRolesDataTablesList = new ArrayList( usersToPersist.Count );

			foreach( VbecsUser user in usersToPersist )
			{
				if( user.DoesVbecsUserCoreRequireSave )
					usersCoreData.ImportRow( user.ExportObjectCoreToDataRow() );

				if( user.DivisionsAndRoles.IsDirty )
					usersRolesDataTablesList.Add( user.DivisionsAndRoles.ExportToSaveDataTable( user ) );
			}
		}

		/// <summary>
		/// ExportObjectCoreToDataRow
		/// </summary>
		/// <returns></returns>
		protected DataRow ExportObjectCoreToDataRow()
		{
			DataTable dtUserCoreData = GetUserTemplateDataTable();
			dtUserCoreData.Rows.Add( dtUserCoreData.NewRow() );

			LoadDataRowFromThis( dtUserCoreData.Rows[0] );

			return dtUserCoreData.Rows[0];
		}

		/// <summary>
		/// Sets broken rules based on <see cref="UniqueDatabaseConstraintViolatedException"/>. Basically, 
		/// converts conflicting column name to broken rule. Indicates if column name is recognized. 
		/// </summary>
		/// <param name="xcp">Exception to analyze.</param>
		/// <returns>True if column name is recognized - at least one appropriate BR is broken. Otherwise - false.</returns>
		protected bool SetPostSaveBrokenRulesBasedOnException( Common.UniqueDatabaseConstraintViolatedException xcp )
		{
			return !Utility.ForcedCheckLogicAnd(
				!RuleBroken( BrRules.UserNtLoginIsNotUnique, xcp.ConflictingDbColumnName == VBECSUSER.VbecsUserId ),
				!RuleBroken( BrRules.UserDuzIsNotUnique, xcp.ConflictingDbColumnName == VBECSUSER.UserDUZ ),
				!RuleBroken( BrRules.UserInitialsAreNotUnique, xcp.ConflictingDbColumnName == VBECSUSER.UserInitials ),
				!RuleBroken( BrRules.VistaUserNameIsNotUnique, xcp.ConflictingDbColumnName == VBECSUSER.VistaUserName ) );
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6956"> 
		///		<ExpectedInput>Current user executing unit tests has access to VBECS.</ExpectedInput>
		///		<ExpectedOutput>No exception is thrown, new user object with matching NT login ID is returned.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6957"> 
		///		<ExpectedInput>Current user is inactivated in the VBECS DB.</ExpectedInput>
		///		<ExpectedOutput>VbecsLogonException.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Logs current Windows application user into the VBECS by checking pre-logon requirements and user access. 
		/// </summary>
		/// <returns>New instance of the <see cref="VbecsUser"/> logged on to VBECS.</returns>
		public static VbecsUser LogonToVbecs()
		{
			DataRow _userDr = DAL.VbecsUser.GetCheckForVbecsUser( LogonUser.LogonUserName );

		    if (_userDr == null)
		    {
                PerformWindowsRequirementsLogonCheck();     //Defect 267675. This method will throw exceptions if the AD Group isn't found or the user
                                                            //is not a member of the Group. If those are both OK, then the user isn't in the DB
                                                            //and allow the following exception (user not in DB) to spew
		        throw (new VbecsLogonException(StrRes.SysErrMsg.UC018.UserNotFound().ResString));
		    }

		    VbecsUser _user = new VbecsUser( _userDr );
			
			PerformUserAccessLogonCheck( _user );

			Common.Utility.EnableSound();

			return _user;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/15/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7969"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7970"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// LogonToVbecsAdministrator
		/// </summary>
		public static void LogonToVbecsAdministrator()
		{
			if( !VbecsWindowsUser.DoesVbecsDomainGroupExist(true) )
				throw( new VbecsLogonException( StrRes.SysErrMsg.UC018.VbecsWindowsGroupDoesNotExist( Common.VbecsConfig.Current.VbecsAdminDomainGroupName ).ResString ) );

			if( !VbecsWindowsUser.IsCurrentUserAdministrator() )
				throw( new VbecsLogonException( Common.StrRes.SysErrMsg.MUC01.UserIsNotAdminAndNotAuthorized().ResString ) );
		}

		/// <summary>
		/// Verifies that all conditions required for logon are met 
		/// (VBECS users Windows group exists, user is a member of the group...). 
		/// </summary>
		protected static void PerformWindowsRequirementsLogonCheck()
		{
			if( !VbecsWindowsUser.DoesVbecsDomainGroupExist(false) )
				throw( new VbecsLogonException( StrRes.SysErrMsg.UC018.VbecsWindowsGroupDoesNotExist( Common.VbecsConfig.Current.VbecsUserDomainGroupName ).ResString ) );

			if( !VbecsWindowsUser.IsCurrentUserVbecsUser() )
				throw( new VbecsLogonException( StrRes.SysErrMsg.UC018.UserIsNotAMemberVbecsGroup( Common.VbecsConfig.Current.VbecsUserDomainGroupName ).ResString ) );
		}

		/// <summary>
		/// Verifies that user is authorized to access VBECS - user record is active and 
		/// user has active division roles in active divisions. 
		/// </summary>
		/// <param name="userToCheck">VBECS user to perform logon check for.</param>
		protected static void PerformUserAccessLogonCheck( VbecsUser userToCheck )
		{
			if( userToCheck == null )
				throw( new ArgumentNullException( "userToCheck" ) );

			if( !userToCheck.IsActive ) 
				throw( new VbecsLogonException( StrRes.SysErrMsg.UC018.UserRecordInactivated().ResString ) );

			if( userToCheck.DivisionsAndRoles.Count < 1 )
				throw( new VbecsLogonException( StrRes.SysErrMsg.UC018.NoDivisionsConfigured().ResString ) );

			if( userToCheck.DivisionsAndRoles.EffectiveDivisions.Count < 1 )
				throw( new VbecsLogonException( StrRes.SysErrMsg.UC018.NoActiveDivisionRolesConfigured().ResString ) );
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/15/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6681"> 
		///		<ExpectedInput>Valid VBECS user and valid division code that user has access to; logon user properties are set to the current.</ExpectedInput>
		///		<ExpectedOutput>No exception is thrown; user's current division corresponds to the specified code.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6682"> 
		///		<ExpectedInput>Invalid (non-existent) division code.</ExpectedInput>
		///		<ExpectedOutput>VbecsLogonException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Logs on current VBECS user to the division with the given code.
		/// </summary>
		/// <param name="divisionCode">Division code for the division to logon to.</param>
		public void LogonToDivision( string divisionCode )
		{
			LoadUserDivisionRoles();
			LoadUserSystemWidePermissions();

			_currentDivisionRole = DivisionsAndRoles[ divisionCode ];

			// **** 508 Compliance ***
			Common.LogonUser.IsVisuallyImpairedUser = _currentDivisionRole.Role.UserRoleId.ToString().IndexOf("VisuallyImpaired") != -1;

			if( _currentDivisionRole == null )
				throw( new VbecsLogonException( StrRes.SysErrMsg.UC018.UserIsNotAuthorizedToAccessDivisionOrItDoesNotExist( divisionCode ).ResString ) );
			
			if( !_currentDivisionRole.IsActive )
				throw( new VbecsLogonException( StrRes.SysErrMsg.UC018.UserDivisionRoleInactivated( _currentDivisionRole.Division.DivisionName ).ResString ) );

			lock( typeof( VbecsUser ) )
			{
				_loggedOnUser = this;

				LogonUser.LogonUserDUZ = this.UserDuz;
				LogonUser.LogonUserDivisionCode = this.CurrentDivisionRole.Division.DivisionCode;
				LogonUser.NotDefined = false;
			}
		}

		///<Developers>
		///	<Developer>VHA</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/23/2007</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8512"> 
		///		<ExpectedInput>System sets to a valid printer (exists on the server).</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8513"> 
		///		<ExpectedInput>System sets to an invalid printer (does not exist on the server).</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// Sets the default printer for the user. CR2320
		///</summary>
		public bool SetDefaultPrinter()
		{
			if (this.CurrentDivisionRole.Division.PrinterName != null && this.CurrentDivisionRole.Division.PrinterName.Trim().Length > 0)
			{
				string path = "win32_printer.DeviceId='" + this.CurrentDivisionRole.Division.PrinterName + "'";
				using (ManagementObject printer = new ManagementObject(path))
				{
					try
					{
						ManagementBaseObject outParams =
							printer.InvokeMethod("SetDefaultPrinter",
							null, null);
						return (int)(uint)outParams.Properties["ReturnValue"].Value == 0;
					}
					// This exception is thrown when VBECS attempts to set a printer that 
					// doesn't exist on the server.  This can happen when a printer has
					// been deleted or renamed.
					catch (System.Management.ManagementException)
					{
						return false;
					}
				}
			}
			else
			{
				return false;
			}
		}


		/// <summary>
		/// Verify that the newly configured division is active in VistA.
		/// </summary>
		/// <returns></returns>
		// CR1902
		// CR2597: removed code that checked to see if DivisionAndRoles was dirty
		private bool VerifyUsersDivisions()
		{
			VbecsVistAUser vistAUser = (VbecsVistAUser) VbecsVistAUser.GetAllVistAUsersHashedByDuz()[this._duz];
						
			for (int i=0; i<this.DivisionsAndRoles.EffectiveDivisions.Count; i++)
			{
				string divisionCode =  this.DivisionsAndRoles.EffectiveDivisions[i].DivisionCode;
				if (!vistAUser.AvailableDivisionsHashedByCode.ContainsKey(divisionCode))
				{
					return false;
				}
			}
			return true;
		}
		// CR1902 end


		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6556"> 
		///		<ExpectedInput>User is logged on to division.</ExpectedInput>
		///		<ExpectedOutput>User that is logged on to division.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6557"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Currently logged on VBECS user. 
		/// </summary>
		public static VbecsUser LoggedOnUser
		{
			get
			{
				lock( typeof(VbecsUser) )
					return _loggedOnUser;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/20/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5003"> 
		///		<ExpectedInput>Valid unit test division code.</ExpectedInput>
		///		<ExpectedOutput>Non-null, non-empty list containing only active users with active division role for the specified division.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5004"> 
		///		<ExpectedInput>Null.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6960"> 
		///		<ExpectedInput>Non-existent (00000) division code.</ExpectedInput>
		///		<ExpectedOutput>Empty list.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets active users that have active roles in division with specified code.
		/// </summary>
		/// <param name="divisionCode">Division code.</param>
		/// <returns>List of active users with active role in a given division.</returns>
		public static ArrayList GetActiveDivisionUsers( string divisionCode )
		{
			if( divisionCode == null )
				throw( new ArgumentNullException( "divisionCode" ) );

			return ConvertUsersDataTableToList( DAL.VbecsUser.GetActiveDivisionUsers( divisionCode ) );
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/15/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7971"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7972"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetVbecsUsersWithRoleInGivenDivision
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static IList GetVbecsUsersWithRoleInGivenDivision( string divisionCode )
		{
			if( divisionCode == null )
				throw( new ArgumentNullException( "divisionCode" ) );

			return ConvertUsersDataTableToList( DAL.VbecsUser.GetVbecsUsersWithRoleInGivenDivision( divisionCode ) );
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/20/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5009"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Non-null, non-empty list of VBECS users.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5010"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves list of all VBECS users for all divisions including inactive users.
		/// </summary>
		/// <returns>List of all users in all divisions.</returns>
		public static IList GetAllUsers()
		{
			return ConvertUsersDataTableToList( DAL.VbecsUser.GetAllUsers() );
		}
		
		/// <summary>
		/// Converts the supplied <see cref="DataTable"/> containing VBECS users data 
		/// into a list of <see cref="VbecsUser"/> objects deserialized from the <see cref="DataTable"/>.
		/// </summary>
		/// <param name="dtTable"><see cref="DataTable"/> containing users data.</param>
		/// <returns>List containing <see cref="VbecsUser"/> objects deserialized from the <see cref="DataTable"/>.</returns>
		protected static ArrayList ConvertUsersDataTableToList( DataTable dtTable )
		{
			ArrayList _userList = new ArrayList( dtTable.Rows.Count );

			foreach( DataRow _dtRow in dtTable.Rows )
				_userList.Add( new BOL.VbecsUser( _dtRow ) );

			return _userList;
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6683"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Non-null object ID (user NT login ID).</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6684"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns unique object ID (NT login name ).
		/// </summary>
		/// <returns>Unique object ID (NT login name).</returns>
		public object GetObjectID()
		{	
			return this.UserNTLoginID;
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6685"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Non-null, non-empty list of VbecsUser objects sorted by user NT login ID.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6686"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves list of VBECS users sorted and hashed by ID.
		/// </summary>
		/// <returns>List of VBECS users.</returns>
		public static BusinessObjectSortedHashList GetAllSortedById()
		{
			DataTable _dt = DAL.VbecsUser.GetAllUsers();
			BusinessObjectSortedHashList _list = new BusinessObjectSortedHashList( new VbecsUserIdComparer(), _dt.Rows.Count );

			foreach( DataRow _dr in _dt.Rows )
				_list.Add( new VbecsUser( _dr ) );

			return _list;		
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6972"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6973"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Inactivates all users in the supplied list. 
		/// </summary>
		/// <param name="usersToInactivate">List of users to inactivate.</param>
		/// <param name="lastUpdateFunctionId">UC or calling method.</param>
		public static void InactivateUsers( IList usersToInactivate, UpdateFunction lastUpdateFunctionId )
		{
			foreach( VbecsUser user in usersToInactivate )
				user.IsActive = false;

			PersistUsers( usersToInactivate, lastUpdateFunctionId );
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>04/01/2005</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="7220"> 
		///		<ExpectedInput>Valid user role id.</ExpectedInput>
		///		<ExpectedOutput>Data table of all the roles with the same or less permission than the input role.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7221"> 
		///		<ExpectedInput>Invalid Id</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///	Gets a list of all the user roles at or under the user role ID passed in
		/// </summary>
		/// <param name="userRoleId">User role ID.</param>
		/// <returns><see cref="DataTable"/> containing list of user roles.</returns>
		public static DataTable GetUserRolesList( int userRoleId )
		{
			return DAL.VbecsUser.GetUserRolesList(userRoleId);
		}

		#region Properties

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6963"> 
		///		<ExpectedInput>User is logged on to the unit test division.</ExpectedInput>
		///		<ExpectedOutput>Division role for the unit test division.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6964"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Current division role. Implements BR_10.01 and BR_19.03.
		/// </summary>
		public VbecsUserDivisionRole CurrentDivisionRole
		{
			get
			{
				return _currentDivisionRole;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6965"> 
		///		<ExpectedInput>Valid user that have an access to the VBECS.</ExpectedInput>
		///		<ExpectedOutput>Non-null, non-empty list.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6966"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// User roles in all divisions.
		/// </summary>
		public VbecsUserDivisionRolesCollection DivisionsAndRoles
		{
			get
			{
				if( _userDivisionRoles == null )
					LoadUserDivisionRoles();

				return _userDivisionRoles;
			}
		}

		/// <summary>
		/// <see cref="DataTable"/> containing user permissions in all divisions.
		/// </summary>
		protected DataTable PermissionsTable
		{
			get
			{
				if( _vbecsUserPermissions == null )
					LoadUserSystemWidePermissions();

				return _vbecsUserPermissions;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6961"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Non-empty GUID.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6962"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Unique identifier on the VbecsUser table.
		/// </summary>
		public Guid VbecsUserGuid
		{
			get
			{
				return _guid;
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/23/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="277"> 
		///		<ExpectedInput>Valid NT login ID (up to 30 chars).</ExpectedInput>
		///		<ExpectedOutput>User NT login ID is set to input value; rule is not broken; object marked as dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="2368"> 
		///		<ExpectedInput>Invalid user name (100 chars).</ExpectedInput>
		///		<ExpectedOutput>User NT login ID is set to input value; corresponding rule is broken; object marked as dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2369"> 
		///		<ExpectedInput>Null user NT login ID.</ExpectedInput>
		///		<ExpectedOutput>User NT login ID is reset; corresponding rule is broken; object marked as dirty.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///	<summary>
		///	The system logon identifier for a VBECS user. Implements BR_19.04.
		///	</summary>
		public string UserNTLoginID
		{
			get
			{
				return _ntLoginName;
			}
			set
			{
				if( _ntLoginName == value )
					return;

				_ntLoginName = (value == null) ? null : value.ToUpper();

				RuleBroken( BrRules.UserNtLoginIdInvalid,
					!RuleBroken( BrRules.UserNtLoginIdNotSet, StUiNullConvert.IsZeroString( _ntLoginName ) ) 
					&& _ntLoginName.Length > 30) ;

				RuleBroken( BrRules.UserNtLoginIsNotUnique, false ); // if it was changed it can be valid, but can be validated only by DB.

				OnPropertyChanged();
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/23/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="276"> 
		///		<ExpectedInput>Valid user name (up to 50 chars).</ExpectedInput>
		///		<ExpectedOutput>User name is set to input value; rule is not broken; object marked as dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2360"> 
		///		<ExpectedInput>Invalid user name (100 chars).</ExpectedInput>
		///		<ExpectedOutput>User name is set to input value; corresponding rule is broken; object marked as dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2361"> 
		///		<ExpectedInput>Null user name.</ExpectedInput>
		///		<ExpectedOutput>User name is reset; corresponding rule is broken; object marked as dirty.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///	<summary>
		///	The full name of the user.
		///	</summary>
		public string UserName
		{
			get
			{
				return _ntName;
			}
			set
			{
				if( _ntName == value )
					return;

				_ntName = value;
				RuleBroken( BrRules.UserNameTooLong, 
					!RuleBroken( BrRules.UserNtNameNotSet, StUiNullConvert.IsZeroString( _ntName )  ) 
					&& _ntName.Length > 50 );

				OnPropertyChanged();
			}
		}


		/// <defects>251166</defects>
        /// <summary>
        /// A users name can change in AD. VBECS reads this value when the user is intiially
        /// entered in VBECS, then never again. This real-time check will allow the Admin to
        /// see what current exists in AD. 
        /// </summary>
	    public string UserNameFromAD
	    {
	        get 
            {
                string retValue = string.Empty;

                foreach (VbecsWindowsUser wUser in VbecsWindowsUser.VbecsUsersADList)
                {
                    if (wUser.LoginId == this.UserNTLoginID)
                    {
                        retValue = wUser.FullName;
                        break;
                    }
                }

                return (retValue);
            }
	    }

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1026"> 
		///		<ExpectedInput>Valid VistA user DUZ (up to 9 chars).</ExpectedInput>
		///		<ExpectedOutput>VistA user DUZ is set to input value; rule is not broken; object marked as dirty.</ExpectedOutput>
		///	</Case>
		///		
		///<Case type="1" testid ="2379"> 
		///		<ExpectedInput>Invalid VistA user DUZ (20 chars).</ExpectedInput>
		///		<ExpectedOutput>VistA user DUZ is set to input value; corresponding rule is broken; object marked as dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2380"> 
		///		<ExpectedInput>Null VistA user DUZ.</ExpectedInput>
		///		<ExpectedOutput>VistA user DUZ is reset; corresponding rule is broken; object marked as dirty.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// User DUZ information from VistA (user unique ID in VistA).
		/// </summary>
		public string UserDuz
		{
			get
			{
				return _duz;
			}

			set
			{
				if( _duz == value )
					return;

				_duz = value;

				RuleBroken( BrRules.UserDuzInvalid,
					!RuleBroken( BrRules.UserDuzNotSet, StUiNullConvert.IsZeroString( _duz ) ) 
					&& !Common.RegularExpressions.VistADuz().IsMatch( _duz ) );

				RuleBroken( BrRules.UserDuzIsNotUnique, false ); // if it was changed it can be valid, but can be validated only by DB.

				OnPropertyChanged();
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6950"> 
		///		<ExpectedInput>Valid VistA user name (up to 50 chars).</ExpectedInput>
		///		<ExpectedOutput>VistA user name is set to input value; rule is not broken; object marked as dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6951"> 
		///		<ExpectedInput>Invalid VistA user name (100 chars).</ExpectedInput>
		///		<ExpectedOutput>VistA user name is set to input value; corresponding rule is broken; object marked as dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6952"> 
		///		<ExpectedInput>Null VistA user name.</ExpectedInput>
		///		<ExpectedOutput>VistA user name is reset; corresponding rule is broken; object marked as dirty.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// VistA user name. It stored in the VBECS DB and may not reflect changes occured in VistA. 
		/// </summary>
		public string VistAUserName
		{
			get
			{
				return _vistaName;
			}
			set
			{
				if( _vistaName == value )
					return;

				_vistaName = value;
				RuleBroken( BrRules.VistaUserNameTooLong,
					!RuleBroken( BrRules.VistaUserNameNotSet, StUiNullConvert.IsZeroString( _vistaName ) )
					&& _vistaName.Length > 50 );

				RuleBroken( BrRules.VistaUserNameIsNotUnique, false ); // if it was changed it can be valid, but can be validated only by DB.

				OnPropertyChanged();
			}
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/24/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3064"> 
		///		<ExpectedInput>Valid user e-mail.</ExpectedInput>
		///		<ExpectedOutput>User email set to input value; rule is not broken; object marked as dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3066"> 
		///		<ExpectedInput>Invalid email address.</ExpectedInput>
		///		<ExpectedOutput>User email is set to input value; corresponding rule is broken; object marked as dirty.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// The Email Address of the user	
		/// </summary>
		public string UserEmail
		{
			get
			{
				return _email;
			}
			set
			{
				if( _email == value )
					return;

				_email = value;

				RuleBroken( BrRules.UserEmailInvalid,
					!StUiNullConvert.IsZeroString( _email ) && 
					!Common.RegularExpressions.EmailAddress().IsMatch( _email ) );

				OnPropertyChanged();
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/29/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3543"> 
		///		<ExpectedInput>Valid user initials (up to 3 chars).</ExpectedInput>
		///		<ExpectedOutput>User initials set to input value; rules are not broken; object marked as dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3544"> 
		///		<ExpectedInput>Invalid user initials (7 chars).</ExpectedInput>
		///		<ExpectedOutput>User initials are set to input value; corresponding rule is broken; object marked as dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6949"> 
		///		<ExpectedInput>Null user initials.</ExpectedInput>
		///		<ExpectedOutput>User initials are reset; corresponding rule is broken; object marked as dirty.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Unique initials of the user (derived from VistA, but VBECS 
		/// user initials do not have to match VistA initials). 
		/// </summary>
		public string UserInitials
		{
			get
			{
				return _initials;
			}
			set
			{
				if( _initials == value )
					return;

				_initials = StNullConvert.IsNull( value ) ? value : value.Trim().ToUpper();

				RuleBroken( BrRules.UserInitialsAreTooLong,
					!RuleBroken( BrRules.UserInitialsNotSet, StUiNullConvert.IsZeroString( _initials ) ) 
					&& _initials.Length > 3 );

				RuleBroken( BrRules.UserInitialsAreNotUnique, false ); // if it was changed it can be valid, but can be validated only in DB.

				OnPropertyChanged();
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6953"> 
		///		<ExpectedInput>False - user is inactive.</ExpectedInput>
		///		<ExpectedOutput>User is inactive (false); object marked as dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6955"> 
		///		<ExpectedInput>True - user is active.</ExpectedInput>
		///		<ExpectedOutput>User is active (true); object marked as dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6954"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Defines whether user is active or not. 
		/// </summary>
		public bool IsActive
		{
			get
			{
				return _isActive;
			}
			set
			{
				if( _isActive == value )
					return;

				_isActive = value;

				OnPropertyChanged();
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/15/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7973"> 
		///		<ExpectedInput>Valid boolean valud</ExpectedInput>
		///		<ExpectedOutput>IsDirty property updated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7974"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// IsDirty
		/// </summary>
		public override bool IsDirty
		{
			get
			{
				return base.IsDirty || ( _userDivisionRoles != null && _userDivisionRoles.IsDirty );
			}
			set
			{
				base.IsDirty = value;
			}
		}

		/// <summary>
		/// DoesVbecsUserCoreRequireSave
		/// </summary>
		protected bool DoesVbecsUserCoreRequireSave
		{
			get
			{
				return base.IsDirty || IsNew;
			}
		}

		#endregion 
	}
}